/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECAMERACALIBRATIONINTERFACE_H
#define FLOWENGINECAMERACALIBRATIONINTERFACE_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! Stores a camera calibration
    class CameraCalibrationInterface
    {
        public:

            //! Calibration Adjustment type used in the Structure from Motion procedure
            enum class AdjustmentType : std::size_t
            {
                //! All the parameters are kept fixed during the SfM procedures
                None = 0,

                //! Adjust internal parameters during the SfM procedures (fx, fy, px, py)
                AdjustInternals,

                //! Adjust internal parameters and radial distortion k1, k2, k3
                //! parameters during the SfM procedures (fx, fy, px, py, k1, k2, k3)
                AdjustInternalsAndK1K2K3,

                //! Adjust internal parameters and radial distortion k1, k2, k3
                //! and tangential distortion p1, p2 parameters during the SfM
                //! procedures (fx, fy, px, py, k1, k2, k3, p1, p2 )
                AdjustInternalsAndK1K2K3P1P2,

                //! Adjust internal parameters during the SfM procedures (fx, fy, px, py)
                //! using a prereconstruction step
                AdjustInternalsWithPrereconstructionStep,

                //! Adjust internal parameters and radial distortion k1, k2, k3
                //! parameters during the SfM procedures (fx, fy, px, py, k1, k2, k3)
                //! using a prereconstruction step
                AdjustInternalsAndK1K2K3WithPrereconstructionStep,

                //! Adjust internal parameters and radial distortion k1, k2, k3 and
                //! tangential distortion p1, p2 parameters during the SfM procedures
                //! (fx, fy, px, py, k1, k2, k3, p1, p2 ) using a prereconstruction step
                AdjustInternalsAndK1K2K3P1P2WithPrereconstructionStep,

                //! Adjust internal parameters during the SfM procedures (fx, fy, px, py)
                //! using a autocalibration (no initial parameters required)
                AdjustInternalsWithAutocalibration,

                //! Adjust internal parameters and radial distortion k1, k2, k3 parameters
                //! during the SfM procedures (fx, fy, px, py, k1, k2, k3) using a
                //! autocalibration (no initial parameters required)
                AdjustInternalsAndK1K2K3WithAutocalibration,

                //! Adjust internal parameters and radial distortion k1, k2, k3
                //! and tangential distortion p1, p2 parameters during the SfM
                //! procedures (fx, fy, px, py, k1, k2, k3, p1, p2 ) using a
                //! autocalibration (no initial parameters required)
                AdjustInternalsAndK1K2K3P1P2WithAutocalibration,
            };

        public:

            //! Default virtual destructor
            FLE_DLL virtual ~CameraCalibrationInterface() = default;

        public:

            //! Returns the Brown's model camera radial distortion coefficients organized as [k1 k2 p1 p2 k3].
            //! @param[out] outRadialDistortion a buffer that receives the data
            //!             in row-major order. The buffer must be at least 5
            //!             elements long.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the buffer has been filled with the radial distortion information.
            //!  - Result::InvalidArgument -- if `outRadialDistortion` is an invalid buffer.
            //!  - Result::BufferTooSmall -- if `outRadialDistortion` is not big enough to receive all the radial distortion info.
            //!  - Result::FeatureNotAvailable -- if called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getRadialDistortion( Buffer< double > outRadialDistortion ) const = 0;

            //! Returns the calibration focal lengths in pixel.
            //! @param[out] outFx focal length in pixel with respect to the x image axis
            //! @param[out] outFy focal length in pixel with respect to the y image axis
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the focal lengths are successfully returned.
            //!  - Result::FeatureNotAvailable -- if called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getFocalLength( double &outFx, double &outFy ) const = 0;

            //! Returns the calibration principal point in pixel.
            //! This function is not available in the free version of the SDK.
            //! @param[out] outPx principal point in pixel with respect to the x image axis
            //! @param[out] outPy principal point in pixel with respect to the y image axis
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the principal point was successfully returned.
            //!  - Result::FeatureNotAvailable -- if called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPrincipalPoint( double &outPx, double &outPy ) const = 0;

            //! Returns the calibration skew point in pixel.
            //! @param[out] outSkew the skew of the calibration.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the value was successfully returned.
            //!  - Result::FeatureNotAvailable -- if called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getSkew( double &outSkew ) const = 0;

            //! Returns the calibration adjustment type
            //! @param[out] outAdj Adjustment type for this calibration
            FLE_DLL virtual void getAdjustmentType( AdjustmentType &outAdj ) const = 0;

        public:

            //! Set the Brown's model camera radial distortion coefficients organized as [k1 k2 p1 p2 k3]
            //! @param[in] radialDistortion pointer to a buffer that receives the data
            //!             in row-major order. The buffer must be at least 5
            //!             elements long or the behavior is undefined
            FLE_DLL virtual Result setRadialDistortion( ConstBuffer< double > radialDistortion ) = 0;

            //! Set the calibration focal lengths in pixel
            //! @param[in] inFx focal length in pixel with respect to the x image axis
            //! @param[in] inFy focal length in pixel with respect to the y image axis
            FLE_DLL virtual void setFocalLength( double inFx, double inFy ) = 0;

            //! Set the calibration principal point in pixel
            //! @param[in] inPx principal point in pixel with respect to the x image axis
            //! @param[in] inPy principal point in pixel with respect to the y image axis
            FLE_DLL virtual void setPrincipalPoint( double inPx, double inPy ) = 0;

            //! Set the calibration skew point in pixel
            //! @param[in] inSkew principal point in pixel with respect to the x image axis
            FLE_DLL virtual void setSkew( double inSkew ) = 0;

            //! Set the calibration adjustment type
            //! @param[in] inAdj Adjustment type for this calibration
            FLE_DLL virtual void setAdjustmentType( const AdjustmentType &inAdj ) = 0;

        public:

            //! Set to adjust or not the focal length during the SfM reconstruction
            //! @param[in] inAdj Adjustment enabled or disabled
            FLE_DLL virtual void setAdjustFocalLength( bool inAdj ) = 0;

            //! Set to adjust or not the principal point during the SfM reconstruction
            //! @param[in] inAdj Adjustment enabled or disabled
            FLE_DLL virtual void setAdjustPrincipalPoint( bool inAdj ) = 0;

            //! Set to adjust or not the radial distortion during the SfM reconstruction
            //! @param[in] inAdj Adjustment enabled or disabled
            FLE_DLL virtual void setAdjustRadialDistortion( bool inAdj ) = 0;

            //! Set to adjust or not the tangential distortion during the SfM reconstruction
            //! @param[in] inAdj Adjustment enabled or disabled
            FLE_DLL virtual void setAdjustTangentialDistortion( bool inAdj ) = 0;

            //! Set to use the precalibration step during the SfM reconstruction.
            //! When disabled the parameters are only adjusted in the final bundle adjustment.
            //! @param[in] inPC Precalibration enabled or disabled
            FLE_DLL virtual void setUsePrecalibrationStep( bool inPC ) = 0;

            //! Set to consider the calibration fully autocalibrated.
            //! @param[in] inAC Autocalibration enabled or disabled
            FLE_DLL virtual void setAutocalibrated( bool inAC ) = 0;

            //! Get the focal length adjustment flag.
            //! @returns Focal length adjustment flag.
            FLE_DLL virtual bool getAdjustFocalLength() const = 0;

            //! Get the principal point adjustment flag.
            //! @returns Principal point adjustment flag.
            FLE_DLL virtual bool getAdjustPrincipalPoint() const = 0;

            //! Get the radial distortion adjustment flag.
            //! @returns Radial distortion adjustment flag.
            FLE_DLL virtual bool getAdjustRadialDistortion() const = 0;

            //! Get the tangential distortion adjustment flag.
            //! @returns Tangential distortion adjustment flag.
            FLE_DLL virtual bool getAdjustTangentialDistortion() const = 0;

            //! Get the precalibration step flag.
            //! @returns Precalibration flag value.
            FLE_DLL virtual bool getUsePrecalibrationStep() const = 0;

            //! Get the autocalibration flag.
            //! @returns Autocalibration flag value.
            FLE_DLL virtual bool getAutocalibrated() const = 0;

        public:

            //! Undistorts the supplied image.
            //! @param[in, out] inOutImage the image to be undistorted.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the image was succesfully undistorted.
            //!  - Result::InvalidArgument -- if `inOutImage` has invalid dimensions or its data buffer is not valid.
            //!  - Result::BufferTooSmall -- if `inOutImage` data buffer is not big enough.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result undistortImage( Image &inOutImage ) const = 0;

        public:

            //! Saves the camera calibration info to an XML file.
            //! @param[in] filePath string buffer that contains the UTF-8 file path.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the operation was successful.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::DiskWriteError -- if the saving process failed for file system related reasons.
            //! - Result::FeatureNotAvailable -- if this method is called by the Free version of the SDK.
            //! - Result::GenericError -- if any other error occurred.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveToXML( ConstStringBuffer filePath ) const = 0;

            //! Loads the camera calibration info from an XML file.
            //! The file shall contain 1 calibration.
            //! @param[in] filePath string buffer that contains the UTF-8 file path.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the operation was successful.
            //! - Result::InvalidArgument --
            //!  - if `filePath` is not a valid string buffer
            //!  - if the file is not a valid calibration XML format
            //!  - if the file contains more than 1 calibration
            //! - Result::FeatureNotAvailable -- if this method is called by the Free version of the SDK.
            //! - Result::GenericError -- if any other error occurred.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result loadFromXML( ConstStringBuffer filePath ) = 0;

        public:

            //! @returns the length of the camera model name of this entity.
            FLE_DLL virtual Size getModelNameLength() const = 0;

            //! @returns the length of the camera model name of this entity.
            //! @param outName a buffer that receives the camera model name of the entity. Must be at least getModelNameLength() long.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the buffer was successfully filled with the entity name.
            //! - Result::InvalidArgument -- if `outName` is not a valid string buffer.
            //! - Result::BufferTooSmall -- if `outName` is not big enough.
            FLE_DLL virtual Result getModelName( StringBuffer outName ) const = 0;

            //! Set the camera model name of this entity.
            //! @param name a constant string buffer containing the camera model name of the entity
            //! @returns One of the following result codes:
            //! - Result::Success -- if the entity name was successfully changed.
            //! - Result::InvalidArgument -- if `name` is not a valid string buffer.
            FLE_DLL virtual Result setModelName( ConstStringBuffer name ) = 0;

            //! @returns the length of the camera make name of this entity.
            FLE_DLL virtual Size getMakeNameLength() const = 0;

            //! @returns the length of the camera make name of this entity.
            //! @param outName a buffer that receives the camera make name of the entity. Must be at least getMakeNameLength() long.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the buffer was successfully filled with the entity name.
            //! - Result::InvalidArgument -- if `outName` is not a valid string buffer.
            //! - Result::BufferTooSmall -- if `outName` is not big enough.
            FLE_DLL virtual Result getMakeName( StringBuffer outName ) const = 0;

            //! Set the camera make name of this entity.
            //! @param name a constant string buffer containing the camera make name of the entity
            //! @returns One of the following result codes:
            //! - Result::Success -- if the entity name was successfully changed.
            //! - Result::InvalidArgument -- if `name` is not a valid string buffer.
            FLE_DLL virtual Result setMakeName( ConstStringBuffer name ) = 0;

            //! @returns the length of the camera lens name of this entity.
            FLE_DLL virtual Size getLensNameLength() const = 0;

            //! @returns the length of the camera lens name of this entity.
            //! @param outName a buffer that receives the name of the entity. Must be at least getLensNameLength() long.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the buffer was successfully filled with the entity name.
            //! - Result::InvalidArgument -- if `outName` is not a valid string buffer.
            //! - Result::BufferTooSmall -- if `outName` is not big enough.
            FLE_DLL virtual Result getLensName( StringBuffer outName ) const = 0;

            //! Set the camera lens name of this entity.
            //! @param name a constant string buffer containing the camera lens name of the entity
            //! @returns One of the following result codes:
            //! - Result::Success -- if the entity name was successfully changed.
            //! - Result::InvalidArgument -- if `name` is not a valid string buffer.
            FLE_DLL virtual Result setLensName( ConstStringBuffer name ) = 0;
    };

    //! Creates a CameraCalibration object
    //! @returns a new CameraCalibration object or nullptr if the operation failed
    FLOWENGINE_FACTORY CameraCalibrationInterface *CreateCameraCalibration();

    //! Destroys a CameraCalibration object
    //! @param[in,out] cameraCalibration pointer to a CameraCalibration created with CreateCameraCalibration()
    FLOWENGINE_FACTORY void DestroyCameraCalibration( CameraCalibrationInterface *cameraCalibration );
}

#endif
